import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";
import { z } from "zod";

const QueryParams = z.object({
  technicianId: z.string().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method !== "GET") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  const parsed = QueryParams.safeParse(req.query);
  if (!parsed.success) {
    return res.status(400).json({
      ok: false,
      error: "Invalid query parameters",
      details: parsed.error.flatten(),
    });
  }

  const { technicianId, startDate, endDate } = parsed.data;

  try {
    // Build where clause for tickets
    const whereClause: any = {
      assignedToId: { not: null },
      startTime: { not: null },
      endTime: { not: null },
      deletedAt: null,
    };

    if (technicianId) {
      whereClause.assignedToId = technicianId;
    }

    if (startDate) {
      whereClause.startTime = {
        ...whereClause.startTime,
        gte: new Date(startDate),
      };
    }

    if (endDate) {
      whereClause.endTime = { ...whereClause.endTime, lte: new Date(endDate) };
    }

    // Get completed tickets with technician info
    const tickets = await prisma.ticket.findMany({
      where: whereClause,
      include: {
        assignedTo: {
          select: {
            id: true,
            name: true,
            email: true,
            hourlyRate: true,
            isExternal: true,
            companyName: true,
          },
        },
      },
      orderBy: { endTime: "desc" },
    });

    // Calculate costs for each ticket
    const ticketsWithCosts = tickets.map((ticket) => {
      if (
        !ticket.startTime ||
        !ticket.endTime ||
        !ticket.assignedTo?.hourlyRate
      ) {
        return {
          ...ticket,
          hoursWorked: 0,
          cost: 0,
        };
      }

      const hoursWorked =
        (ticket.endTime.getTime() - ticket.startTime.getTime()) /
        (1000 * 60 * 60);
      const cost = hoursWorked * ticket.assignedTo.hourlyRate;

      return {
        ...ticket,
        hoursWorked: Math.round(hoursWorked * 100) / 100, // Round to 2 decimal places
        cost: Math.round(cost * 100) / 100, // Round to 2 decimal places
      };
    });

    // Group by technician for summary
    const technicianSummary = ticketsWithCosts.reduce((acc, ticket) => {
      const techId = ticket.assignedToId!;
      const techName = ticket.assignedTo?.name || "Unknown";
      const techEmail = ticket.assignedTo?.email || "";
      const hourlyRate = ticket.assignedTo?.hourlyRate || 0;
      const isExternal = ticket.assignedTo?.isExternal || false;
      const companyName = ticket.assignedTo?.companyName || null;

      if (!acc[techId]) {
        acc[techId] = {
          technicianId: techId,
          technicianName: techName,
          technicianEmail: techEmail,
          hourlyRate,
          isExternal,
          companyName,
          totalTickets: 0,
          totalHours: 0,
          totalCost: 0,
          tickets: [],
        };
      }

      acc[techId].totalTickets += 1;
      acc[techId].totalHours += ticket.hoursWorked;
      acc[techId].totalCost += ticket.cost;
      acc[techId].tickets.push({
        ticketId: ticket.id,
        title: ticket.title,
        startTime: ticket.startTime,
        endTime: ticket.endTime,
        hoursWorked: ticket.hoursWorked,
        cost: ticket.cost,
      });

      return acc;
    }, {} as Record<string, any>);

    // Calculate overall totals
    const totalTickets = ticketsWithCosts.length;
    const totalHours = ticketsWithCosts.reduce(
      (sum, ticket) => sum + ticket.hoursWorked,
      0
    );
    const totalCost = ticketsWithCosts.reduce(
      (sum, ticket) => sum + ticket.cost,
      0
    );

    const summary = {
      totalTickets,
      totalHours: Math.round(totalHours * 100) / 100,
      totalCost: Math.round(totalCost * 100) / 100,
      technicianCount: Object.keys(technicianSummary).length,
    };

    return res.status(200).json({
      ok: true,
      summary,
      technicians: Object.values(technicianSummary),
      tickets: ticketsWithCosts,
    });
  } catch (error) {
    console.error("Error fetching cost analysis:", error);
    return res
      .status(500)
      .json({ ok: false, error: "Failed to fetch cost analysis" });
  }
}
